;+
; NAME:
;   TDB2TDT
;
; AUTHOR:
;   Craig B. Markwardt, NASA/GSFC Code 662, Greenbelt, MD 20770
;   craigm@lheamail.gsfc.nasa.gov
;   UPDATED VERSIONs can be found on my WEB PAGE: 
;      http://cow.physics.wisc.edu/~craigm/idl/idl.html
;
; PURPOSE:
;   Relativistic clock corrections due to Earth motion in solar system 
;
; MAJOR TOPICS:
;   Planetary Orbits
;
; CALLING SEQUENCE:
;   corr = TDB2TDT(JD, TBASE=, DERIV=deriv)
;
; DESCRIPTION:
;
;   The function TDB2TDT computes relativistic corrections that must
;   be applied when performing high precision absolute timing in the
;   solar system.
;
;   According to general relativity, moving clocks, and clocks at
;   different gravitational potentials, will run at different rates
;   with respect to each other.  A clock placed on the earth will run
;   at a time-variable rate because of the non-constant influence of
;   the sun and other planets.  Thus, for the most demanding
;   astrophysical timing applications -- high precision pulsar timing
;   -- times in the accelerating earth observer's frame must be
;   corrected to an inertial frame, such as the solar system
;   barycenter (SSB).  This correction is also convenient because the
;   coordinate time at the SSB is the ephemeris time of the JPL
;   Planetary Ephemeris.
;
;   In general, the difference in the rate of Ti, the time kept by an
;   arbitrary clock, and the rate of T, the ephemeris time, is given
;   by the expression (Standish 1998):
;
;      dTi/dT = 1 - (Ui + vi^2/2) / c^2
;
;   where Ui is the potential of clock i, and vi is the velocity of
;   clock i.  However, when integrated, this expression depends on the
;   position of an individual clock.  A more convenient approximate
;   expression is:
;
;     T = Ti + (robs(Ti) . vearth(T))/c^2 + dtgeo(Ti) + TDB2TDT(Ti)
;
;   where robs is the vector from the geocenter to the observer;
;   vearth is the vector velocity of the earth; and dtgeo is a
;   correction to convert from the observer's clock to geocentric TT
;   time.  TDB2TDT is the value computed by this function, the
;   correction to convert from the geocenter to the solar system
;   barycenter.
;
;   As the above equation shows, while this function provides an
;   important component of the correction, the user must also be
;   responsible for (a) correcting their times to the geocenter (ie,
;   by maintaining atomic clock corrections); (b) estimating the
;   observatory position vector; and and (c) estimating earth's
;   velocity vector (using JPLEPHINTERP).
;
;   Users may note a circularity to the above equation, since
;   vearth(T) is expressed in terms of the SSB coordinate time.  This
;   appears to be a chicken and egg problem since in order to get the
;   earth's velocity, the ephemeris time is needed to begin with.
;   However, to the precision of the above equation, < 25 ns, it is
;   acceptable to replace vearth(T) with vearth(TT).
;
;   The method of computation of TDB2TDT in this function is based on
;   the analytical formulation by Fairhead, Bretagnon & Lestrade, 1988
;   (so-called FBL model) and Fairhead & Bretagnon 1990, in terms of
;   sinusoids of various amplitudes.  TDB2TDT has a dominant periodic
;   component of period 1 year and amplitude 1.7 ms.  The set of 791
;   coefficients used here were drawn from the Princeton pulsar timing
;   program TEMPO version 11.005 (Taylor & Weisberg 1989).
;
;   Because the TDB2TDT quantity is rather expensive to compute but
;   slowly varying, users may wish to also retrieve the time
;   derivative using the DERIV keyword, if they have many times to
;   convert over a short baseline.
;
; Verification
;
;   This implementation has been compared against a set of FBL test
;   data found in the 1996 IERS Conventions, Chapter 11, provided by
;   T. Fukushima.  It has been verified that this routine reproduces
;   the Fukushima numbers to the accuracy of the table, within
;   10^{-14} seconds.
;
;   Fukushima (1995) has found that the 791-term Fairhead & Bretagnon
;   analytical approximation use here has a maximum error of 23
;   nanoseconds in the time range 1980-2000, compared to a numerical
;   integration.  In comparison the truncated 127-term approximation
;   has an error of ~130 nanoseconds.
;
;
; PARAMETERS: 
;
;   JD - Geocentric time TT, scalar or vector, expressed in Julian
;        days.  The actual time used is (JD + TBASE).  For maximum
;        precision, TBASE should be used to express a fixed epoch in
;        whole day numbers, and JD should express fractional offset
;        days from that epoch.
;
;
; KEYWORD PARAMETERS:
;
;   TBASE - scalar Julian day of a fixed epoch, which provides the
;           origin for times passed in JD.
;          Default: 0
;
;   DERIV - upon return, contains the derivative of TDB2TDT in units
;           of seconds per day.  As many derivatives are returned as
;           values passed in JD.
;
;
; RETURNS:
;   The correction offset(s) in units of seconds, to be applied as
;   noted above.
;
;
; EXAMPLE:
;
;   Find the correction at ephemeris time 2451544.5 (JD):
;     IDL> print, tdb2tdt(2451544.5d)
;       -0.00011376314
;   or 0.11 ms.
;
;
; REFERENCES:
;
;   Princeton TEMPO Program
;      http://pulsar.princeton.edu/tempo/
;
;   FBL Test Data Set
;      ftp://maia.usno.navy.mil/conventions/chapter11/fbl.results
;
;   Fairhead, L. & Bretagnon, P. 1990, A&A, 229, 240
;     (basis of this routine)
;
;   Fairhead, L. Bretagnon, P. & Lestrade, J.-F. 1988, in *The Earth's
;     Rotation and Reference Frames for Geodesy and Geodynamics*,
;     ed. A. K. Babcock and G. A. Wilkins, (Dordrecht: Kluwer), p. 419
;     (original "FBL" paper)
;
;   Fukushima, T. 1995, A&A, 294, 895  (error analysis)
;
;   Irwin, A. W. & Fukushima, T. 1999, A&A, 348, 642  (error analysis)
;
;   Standish, E. M. 1998, A&A, 336, 381 (description of time scales)
;
;   Taylor, J. H. & Weisberg, J. M. 1989, ApJ, 345, 434 (pulsar timing)
;
;
; SEE ALSO
;   JPLEPHREAD, JPLEPHINTERP, JPLEPHTEST
;   
; MODIFICATION HISTORY:
;   Original logic from Fairhead & Bretagnon, 1990
;   Drawn from TEMPO v. 11.005, copied 20 Jun 2001
;   Documented and vectorized, 30 Jun 2001
;   
;
;  $Id: tdb2tdt.pro,v 1.4 2001/07/01 07:37:40 craigm Exp $
;
;-
; Copyright (C) 2001, Craig Markwardt
; This software is provided as is without any warranty whatsoever.
; Permission to use, copy and distribute unmodified copies for
; non-commercial purposes, and to modify and use for personal or
; internal use, is granted.  All other rights are reserved.
;-


function tdb2tdt_calc, jd, deriv=deriv, tbase=tbase

  common tdb2tdt_common, const0, freq0, phase0, texp
  if n_elements(const0) EQ 0 then begin
fbldata = [ $
1656.674564d,   6283.075849991d, 6.240054195d, $
  22.417471d,   5753.384884897d, 4.296977442d, $
  13.839792d,  12566.151699983d, 6.196904410d, $
   4.770086d,    529.690965095d, 0.444401603d, $
   4.676740d,   6069.776754553d, 4.021195093d, $
   2.256707d,    213.299095438d, 5.543113262d, $
   1.694205d,     -3.523118349d, 5.025132748d, $
   1.554905d,  77713.771467920d, 5.198467090d, $
   1.276839d,   7860.419392439d, 5.988822341d, $
   1.193379d,   5223.693919802d, 3.649823730d, $
   1.115322d,   3930.209696220d, 1.422745069d, $
   0.794185d,  11506.769769794d, 2.322313077d, $
   0.447061d,     26.298319800d, 3.615796498d, $
   0.435206d,   -398.149003408d, 4.349338347d, $
   0.600309d,   1577.343542448d, 2.678271909d, $
   0.496817d,   6208.294251424d, 5.696701824d, $
   0.486306d,   5884.926846583d, 0.520007179d, $
   0.432392d,     74.781598567d, 2.435898309d, $
   0.468597d,   6244.942814354d, 5.866398759d, $
   0.375510d,   5507.553238667d, 4.103476804d, $
   0.243085d,   -775.522611324d, 3.651837925d, $
   0.173435d,  18849.227549974d, 6.153743485d, $
   0.230685d,   5856.477659115d, 4.773852582d, $
   0.203747d,  12036.460734888d, 4.333987818d, $
   0.143935d,   -796.298006816d, 5.957517795d  ]
fbldata = [ fbldata, $
   0.159080d,  10977.078804699d, 1.890075226d, $
   0.119979d,     38.133035638d, 4.551585768d, $
   0.118971d,   5486.777843175d, 1.914547226d, $
   0.116120d,   1059.381930189d, 0.873504123d, $
   0.137927d,  11790.629088659d, 1.135934669d, $
   0.098358d,   2544.314419883d, 0.092793886d, $
   0.101868d,  -5573.142801634d, 5.984503847d, $
   0.080164d,    206.185548437d, 2.095377709d, $
   0.079645d,   4694.002954708d, 2.949233637d, $
   0.062617d,     20.775395492d, 2.654394814d, $
   0.075019d,   2942.463423292d, 4.980931759d, $
   0.064397d,   5746.271337896d, 1.280308748d, $
   0.063814d,   5760.498431898d, 4.167901731d, $
   0.048042d,   2146.165416475d, 1.495846011d, $
   0.048373d,    155.420399434d, 2.251573730d, $
   0.058844d,    426.598190876d, 4.839650148d, $
   0.046551d,     -0.980321068d, 0.921573539d, $
   0.054139d,  17260.154654690d, 3.411091093d, $
   0.042411d,   6275.962302991d, 2.869567043d, $
   0.040184d,     -7.113547001d, 3.565975565d, $
   0.036564d,   5088.628839767d, 3.324679049d, $
   0.040759d,  12352.852604545d, 3.981496998d, $
   0.036507d,    801.820931124d, 6.248866009d, $
   0.036955d,   3154.687084896d, 5.071801441d, $
   0.042732d,    632.783739313d, 5.720622217d  ]
fbldata = [ fbldata, $
   0.042560d, 161000.685737473d, 1.270837679d, $
   0.040480d,  15720.838784878d, 2.546610123d, $
   0.028244d,  -6286.598968340d, 5.069663519d, $
   0.033477d,   6062.663207553d, 4.144987272d, $
   0.034867d,    522.577418094d, 5.210064075d, $
   0.032438d,   6076.890301554d, 0.749317412d, $
   0.030215d,   7084.896781115d, 3.389610345d, $
   0.029247d, -71430.695617928d, 4.183178762d, $
   0.033529d,   9437.762934887d, 2.404714239d, $
   0.032423d,   8827.390269875d, 5.541473556d, $
   0.027567d,   6279.552731642d, 5.040846034d, $
   0.029862d,  12139.553509107d, 1.770181024d, $
   0.022509d,  10447.387839604d, 1.460726241d, $
   0.020937d,   8429.241266467d, 0.652303414d, $
   0.020322d,    419.484643875d, 3.735430632d, $
   0.024816d,  -1194.447010225d, 1.087136918d, $
   0.025196d,   1748.016413067d, 2.901883301d, $
   0.021691d,  14143.495242431d, 5.952658009d, $
   0.017673d,   6812.766815086d, 3.186129845d, $
   0.022567d,   6133.512652857d, 3.307984806d, $
   0.016155d,  10213.285546211d, 1.331103168d, $
   0.014751d,   1349.867409659d, 4.308933301d, $
   0.015949d,   -220.412642439d, 4.005298270d, $
   0.015974d,  -2352.866153772d, 6.145309371d, $
   0.014223d,  17789.845619785d, 2.104551349d  ]
fbldata = [ fbldata, $
   0.017806d,     73.297125859d, 3.475975097d, $
   0.013671d,   -536.804512095d, 5.971672571d, $
   0.011942d,   8031.092263058d, 2.053414715d, $
   0.014318d,  16730.463689596d, 3.016058075d, $
   0.012462d,    103.092774219d, 1.737438797d, $
   0.010962d,      3.590428652d, 2.196567739d, $
   0.015078d,  19651.048481098d, 3.969480770d, $
   0.010396d,    951.718406251d, 5.717799605d, $
   0.011707d,  -4705.732307544d, 2.654125618d, $
   0.010453d,   5863.591206116d, 1.913704550d, $
   0.012420d,   4690.479836359d, 4.734090399d, $
   0.011847d,   5643.178563677d, 5.489005403d, $
   0.008610d,   3340.612426700d, 3.661698944d, $
   0.011622d,   5120.601145584d, 4.863931876d, $
   0.010825d,    553.569402842d, 0.842715011d, $
   0.008666d,   -135.065080035d, 3.293406547d, $
   0.009963d,    149.563197135d, 4.870690598d, $
   0.009858d,   6309.374169791d, 1.061816410d, $
   0.007959d,    316.391869657d, 2.465042647d, $
   0.010099d,    283.859318865d, 1.942176992d, $
   0.007147d,   -242.728603974d, 3.661486981d, $
   0.007505d,   5230.807466803d, 4.920937029d, $
   0.008323d,  11769.853693166d, 1.229392026d, $
   0.007490d,  -6256.777530192d, 3.658444681d, $
   0.009370d, 149854.400134205d, 0.673880395d  ]
fbldata = [ fbldata, $
   0.007117d,     38.027672636d, 5.294249518d, $
   0.007857d,  12168.002696575d, 0.525733528d, $
   0.007019d,   6206.809778716d, 0.837688810d, $
   0.006056d,    955.599741609d, 4.194535082d, $
   0.008107d,  13367.972631107d, 3.793235253d, $
   0.006731d,   5650.292110678d, 5.639906583d, $
   0.007332d,     36.648562930d, 0.114858677d, $
   0.006366d,   4164.311989613d, 2.262081818d, $
   0.006858d,   5216.580372801d, 0.642063318d, $
   0.006919d,   6681.224853400d, 6.018501522d, $
   0.006826d,   7632.943259650d, 3.458654112d, $
   0.005308d,  -1592.596013633d, 2.500382359d, $
   0.005096d,  11371.704689758d, 2.547107806d, $
   0.004841d,   5333.900241022d, 0.437078094d, $
   0.005582d,   5966.683980335d, 2.246174308d, $
   0.006304d,  11926.254413669d, 2.512929171d, $
   0.006603d,  23581.258177318d, 5.393136889d, $
   0.005123d,     -1.484472708d, 2.999641028d, $
   0.004648d,   1589.072895284d, 1.275847090d, $
   0.005119d,   6438.496249426d, 1.486539246d, $
   0.004521d,   4292.330832950d, 6.140635794d, $
   0.005680d,  23013.539539587d, 4.557814849d, $
   0.005488d,     -3.455808046d, 0.090675389d, $
   0.004193d,   7234.794256242d, 4.869091389d, $
   0.003742d,   7238.675591600d, 4.691976180d  ]
fbldata = [ fbldata, $
   0.004148d,   -110.206321219d, 3.016173439d, $
   0.004553d,  11499.656222793d, 5.554998314d, $
   0.004892d,   5436.993015240d, 1.475415597d, $
   0.004044d,   4732.030627343d, 1.398784824d, $
   0.004164d,  12491.370101415d, 5.650931916d, $
   0.004349d,  11513.883316794d, 2.181745369d, $
   0.003919d,  12528.018664345d, 5.823319737d, $
   0.003129d,   6836.645252834d, 0.003844094d, $
   0.004080d,  -7058.598461315d, 3.690360123d, $
   0.003270d,     76.266071276d, 1.517189902d, $
   0.002954d,   6283.143160294d, 4.447203799d, $
   0.002872d,     28.449187468d, 1.158692983d, $
   0.002881d,    735.876513532d, 0.349250250d, $
   0.003279d,   5849.364112115d, 4.893384368d, $
   0.003625d,   6209.778724132d, 1.473760578d, $
   0.003074d,    949.175608970d, 5.185878737d, $
   0.002775d,   9917.696874510d, 1.030026325d, $
   0.002646d,  10973.555686350d, 3.918259169d, $
   0.002575d,  25132.303399966d, 6.109659023d, $
   0.003500d,    263.083923373d, 1.892100742d, $
   0.002740d,  18319.536584880d, 4.320519510d, $
   0.002464d,    202.253395174d, 4.698203059d, $
   0.002409d,      2.542797281d, 5.325009315d, $
   0.003354d, -90955.551694697d, 1.942656623d, $
   0.002296d,   6496.374945429d, 5.061810696d  ]
fbldata = [ fbldata, $
   0.003002d,   6172.869528772d, 2.797822767d, $
   0.003202d,  27511.467873537d, 0.531673101d, $
   0.002954d,  -6283.008539689d, 4.533471191d, $
   0.002353d,    639.897286314d, 3.734548088d, $
   0.002401d,  16200.772724501d, 2.605547070d, $
   0.003053d, 233141.314403759d, 3.029030662d, $
   0.003024d,  83286.914269554d, 2.355556099d, $
   0.002863d,  17298.182327326d, 5.240963796d, $
   0.002103d,  -7079.373856808d, 5.756641637d, $
   0.002303d,  83996.847317911d, 2.013686814d, $
   0.002303d,  18073.704938650d, 1.089100410d, $
   0.002381d,     63.735898303d, 0.759188178d, $
   0.002493d,   6386.168624210d, 0.645026535d, $
   0.002366d,      3.932153263d, 6.215885448d, $
   0.002169d,  11015.106477335d, 4.845297676d, $
   0.002397d,   6243.458341645d, 3.809290043d, $
   0.002183d,   1162.474704408d, 6.179611691d, $
   0.002353d,   6246.427287062d, 4.781719760d, $
   0.002199d,   -245.831646229d, 5.956152284d, $
   0.001729d,   3894.181829542d, 1.264976635d, $
   0.001896d,  -3128.388765096d, 4.914231596d, $
   0.002085d,     35.164090221d, 1.405158503d, $
   0.002024d,  14712.317116458d, 2.752035928d, $
   0.001737d,   6290.189396992d, 5.280820144d, $
   0.002229d,    491.557929457d, 1.571007057d  ]
fbldata = [ fbldata, $
   0.001602d,  14314.168113050d, 4.203664806d, $
   0.002186d,    454.909366527d, 1.402101526d, $
   0.001897d,  22483.848574493d, 4.167932508d, $
   0.001825d,  -3738.761430108d, 0.545828785d, $
   0.001894d,   1052.268383188d, 5.817167450d, $
   0.001421d,     20.355319399d, 2.419886601d, $
   0.001408d,  10984.192351700d, 2.732084787d, $
   0.001847d,  10873.986030480d, 2.903477885d, $
   0.001391d,  -8635.942003763d, 0.593891500d, $
   0.001388d,     -7.046236698d, 1.166145902d, $
   0.001810d, -88860.057071188d, 0.487355242d, $
   0.001288d,  -1990.745017041d, 3.913022880d, $
   0.001297d,  23543.230504682d, 3.063805171d, $
   0.001335d,   -266.607041722d, 3.995764039d, $
   0.001376d,  10969.965257698d, 5.152914309d, $
   0.001745d, 244287.600007027d, 3.626395673d, $
   0.001649d,  31441.677569757d, 1.952049260d, $
   0.001416d,   9225.539273283d, 4.996408389d, $
   0.001238d,   4804.209275927d, 5.503379738d, $
   0.001472d,   4590.910180489d, 4.164913291d, $
   0.001169d,   6040.347246017d, 5.841719038d, $
   0.001039d,   5540.085789459d, 2.769753519d, $
   0.001004d,   -170.672870619d, 0.755008103d, $
   0.001284d,  10575.406682942d, 5.306538209d, $
   0.001278d,     71.812653151d, 4.713486491d  ]
fbldata = [ fbldata, $
   0.001321d,  18209.330263660d, 2.624866359d, $
   0.001297d,  21228.392023546d, 0.382603541d, $
   0.000954d,   6282.095528923d, 0.882213514d, $
   0.001145d,   6058.731054289d, 1.169483931d, $
   0.000979d,   5547.199336460d, 5.448375984d, $
   0.000987d,  -6262.300454499d, 2.656486959d, $
   0.001070d,-154717.609887482d, 1.827624012d, $
   0.000991d,   4701.116501708d, 4.387001801d, $
   0.001155d,    -14.227094002d, 3.042700750d, $
   0.001176d,    277.034993741d, 3.335519004d, $
   0.000890d,  13916.019109642d, 5.601498297d, $
   0.000884d,  -1551.045222648d, 1.088831705d, $
   0.000876d,   5017.508371365d, 3.969902609d, $
   0.000806d,  15110.466119866d, 5.142876744d, $
   0.000773d,  -4136.910433516d, 0.022067765d, $
   0.001077d,    175.166059800d, 1.844913056d, $
   0.000954d,  -6284.056171060d, 0.968480906d, $
   0.000737d,   5326.786694021d, 4.923831588d, $
   0.000845d,   -433.711737877d, 4.749245231d, $
   0.000819d,   8662.240323563d, 5.991247817d, $
   0.000852d,    199.072001436d, 2.189604979d, $
   0.000723d,  17256.631536341d, 6.068719637d, $
   0.000940d,   6037.244203762d, 6.197428148d, $
   0.000885d,  11712.955318231d, 3.280414875d, $
   0.000706d,  12559.038152982d, 2.824848947d  ]
fbldata = [ fbldata, $
   0.000732d,   2379.164473572d, 2.501813417d, $
   0.000764d,  -6127.655450557d, 2.236346329d, $
   0.000908d,    131.541961686d, 2.521257490d, $
   0.000907d,  35371.887265976d, 3.370195967d, $
   0.000673d,   1066.495477190d, 3.876512374d, $
   0.000814d,  17654.780539750d, 4.627122566d, $
   0.000630d,     36.027866677d, 0.156368499d, $
   0.000798d,    515.463871093d, 5.151962502d, $
   0.000798d,    148.078724426d, 5.909225055d, $
   0.000806d,    309.278322656d, 6.054064447d, $
   0.000607d,    -39.617508346d, 2.839021623d, $
   0.000601d,    412.371096874d, 3.984225404d, $
   0.000646d,  11403.676995575d, 3.852959484d, $
   0.000704d,  13521.751441591d, 2.300991267d, $
   0.000603d, -65147.619767937d, 4.140083146d, $
   0.000609d,  10177.257679534d, 0.437122327d, $
   0.000631d,   5767.611978898d, 4.026532329d, $
   0.000576d,  11087.285125918d, 4.760293101d, $
   0.000674d,  14945.316173554d, 6.270510511d, $
   0.000726d,   5429.879468239d, 6.039606892d, $
   0.000710d,  28766.924424484d, 5.672617711d, $
   0.000647d,  11856.218651625d, 3.397132627d, $
   0.000678d,  -5481.254918868d, 6.249666675d, $
   0.000618d,  22003.914634870d, 2.466427018d, $
   0.000738d,   6134.997125565d, 2.242668890d  ]
fbldata = [ fbldata, $
   0.000660d,    625.670192312d, 5.864091907d, $
   0.000694d,   3496.032826134d, 2.668309141d, $
   0.000531d,   6489.261398429d, 1.681888780d, $
   0.000611d,-143571.324284214d, 2.424978312d, $
   0.000575d,  12043.574281889d, 4.216492400d, $
   0.000553d,  12416.588502848d, 4.772158039d, $
   0.000689d,   4686.889407707d, 6.224271088d, $
   0.000495d,   7342.457780181d, 3.817285811d, $
   0.000567d,   3634.621024518d, 1.649264690d, $
   0.000515d,  18635.928454536d, 3.945345892d, $
   0.000486d,   -323.505416657d, 4.061673868d, $
   0.000662d,  25158.601719765d, 1.794058369d, $
   0.000509d,    846.082834751d, 3.053874588d, $
   0.000472d, -12569.674818332d, 5.112133338d, $
   0.000461d,   6179.983075773d, 0.513669325d, $
   0.000641d,  83467.156352816d, 3.210727723d, $
   0.000520d,  10344.295065386d, 2.445597761d, $
   0.000493d,  18422.629359098d, 1.676939306d, $
   0.000478d,   1265.567478626d, 5.487314569d, $
   0.000472d,    -18.159247265d, 1.999707589d, $
   0.000559d,  11190.377900137d, 5.783236356d, $
   0.000494d,   9623.688276691d, 3.022645053d, $
   0.000463d,   5739.157790895d, 1.411223013d, $
   0.000432d,  16858.482532933d, 1.179256434d, $
   0.000574d,  72140.628666286d, 1.758191830d  ]
fbldata = [ fbldata, $
   0.000484d,  17267.268201691d, 3.290589143d, $
   0.000550d,   4907.302050146d, 0.864024298d, $
   0.000399d,     14.977853527d, 2.094441910d, $
   0.000491d,    224.344795702d, 0.878372791d, $
   0.000432d,  20426.571092422d, 6.003829241d, $
   0.000481d,   5749.452731634d, 4.309591964d, $
   0.000480d,   5757.317038160d, 1.142348571d, $
   0.000485d,   6702.560493867d, 0.210580917d, $
   0.000426d,   6055.549660552d, 4.274476529d, $
   0.000480d,   5959.570433334d, 5.031351030d, $
   0.000466d,  12562.628581634d, 4.959581597d, $
   0.000520d,  39302.096962196d, 4.788002889d, $
   0.000458d,  12132.439962106d, 1.880103788d, $
   0.000470d,  12029.347187887d, 1.405611197d, $
   0.000416d,  -7477.522860216d, 1.082356330d, $
   0.000449d,  11609.862544012d, 4.179989585d, $
   0.000465d,  17253.041107690d, 0.353496295d, $
   0.000362d,  -4535.059436924d, 1.583849576d, $
   0.000383d,  21954.157609398d, 3.747376371d, $
   0.000389d,     17.252277143d, 1.395753179d, $
   0.000331d,  18052.929543158d, 0.566790582d, $
   0.000430d,  13517.870106233d, 0.685827538d, $
   0.000368d,  -5756.908003246d, 0.731374317d, $
   0.000330d,  10557.594160824d, 3.710043680d, $
   0.000332d,  20199.094959633d, 1.652901407d  ]
fbldata = [ fbldata, $
   0.000384d,  11933.367960670d, 5.827781531d, $
   0.000387d,  10454.501386605d, 2.541182564d, $
   0.000325d,  15671.081759407d, 2.178850542d, $
   0.000318d,    138.517496871d, 2.253253037d, $
   0.000305d,   9388.005909415d, 0.578340206d, $
   0.000352d,   5749.861766548d, 3.000297967d, $
   0.000311d,   6915.859589305d, 1.693574249d, $
   0.000297d,  24072.921469776d, 1.997249392d, $
   0.000363d,   -640.877607382d, 5.071820966d, $
   0.000323d,  12592.450019783d, 1.072262823d, $
   0.000341d,  12146.667056108d, 4.700657997d, $
   0.000290d,   9779.108676125d, 1.812320441d, $
   0.000342d,   6132.028180148d, 4.322238614d, $
   0.000329d,   6268.848755990d, 3.033827743d, $
   0.000374d,  17996.031168222d, 3.388716544d, $
   0.000285d,   -533.214083444d, 4.687313233d, $
   0.000338d,   6065.844601290d, 0.877776108d, $
   0.000276d,     24.298513841d, 0.770299429d, $
   0.000336d,  -2388.894020449d, 5.353796034d, $
   0.000290d,   3097.883822726d, 4.075291557d, $
   0.000318d,    709.933048357d, 5.941207518d, $
   0.000271d,  13095.842665077d, 3.208912203d, $
   0.000331d,   6073.708907816d, 4.007881169d, $
   0.000292d,    742.990060533d, 2.714333592d, $
   0.000362d,  29088.811415985d, 3.215977013d  ]
fbldata = [ fbldata, $
   0.000280d,  12359.966151546d, 0.710872502d, $
   0.000267d,  10440.274292604d, 4.730108488d, $
   0.000262d,    838.969287750d, 1.327720272d, $
   0.000250d,  16496.361396202d, 0.898769761d, $
   0.000325d,  20597.243963041d, 0.180044365d, $
   0.000268d,   6148.010769956d, 5.152666276d, $
   0.000284d,   5636.065016677d, 5.655385808d, $
   0.000301d,   6080.822454817d, 2.135396205d, $
   0.000294d,   -377.373607916d, 3.708784168d, $
   0.000236d,   2118.763860378d, 1.733578756d, $
   0.000234d,   5867.523359379d, 5.575209112d, $
   0.000268d,-226858.238553767d, 0.069432392d, $
   0.000265d, 167283.761587465d, 4.369302826d, $
   0.000280d,  28237.233459389d, 5.304829118d, $
   0.000292d,  12345.739057544d, 4.096094132d, $
   0.000223d,  19800.945956225d, 3.069327406d, $
   0.000301d,  43232.306658416d, 6.205311188d, $
   0.000264d,  18875.525869774d, 1.417263408d, $
   0.000304d,  -1823.175188677d, 3.409035232d, $
   0.000301d,    109.945688789d, 0.510922054d, $
   0.000260d,    813.550283960d, 2.389438934d, $
   0.000299d, 316428.228673312d, 5.384595078d, $
   0.000211d,   5756.566278634d, 3.789392838d, $
   0.000209d,   5750.203491159d, 1.661943545d, $
   0.000240d,  12489.885628707d, 5.684549045d  ]
fbldata = [ fbldata, $
   0.000216d,   6303.851245484d, 3.862942261d, $
   0.000203d,   1581.959348283d, 5.549853589d, $
   0.000200d,   5642.198242609d, 1.016115785d, $
   0.000197d,    -70.849445304d, 4.690702525d, $
   0.000227d,   6287.008003254d, 2.911891613d, $
   0.000197d,    533.623118358d, 1.048982898d, $
   0.000205d,  -6279.485421340d, 1.829362730d, $
   0.000209d, -10988.808157535d, 2.636140084d, $
   0.000208d,   -227.526189440d, 4.127883842d, $
   0.000191d,    415.552490612d, 4.401165650d, $
   0.000190d,  29296.615389579d, 4.175658539d, $
   0.000264d,  66567.485864652d, 4.601102551d, $
   0.000256d,  -3646.350377354d, 0.506364778d, $
   0.000188d,  13119.721102825d, 2.032195842d, $
   0.000185d,   -209.366942175d, 4.694756586d, $
   0.000198d,  25934.124331089d, 3.832703118d, $
   0.000195d,   4061.219215394d, 3.308463427d, $
   0.000234d,   5113.487598583d, 1.716090661d, $
   0.000188d,   1478.866574064d, 5.686865780d, $
   0.000222d,  11823.161639450d, 1.942386641d, $
   0.000181d,  10770.893256262d, 1.999482059d, $
   0.000171d,   6546.159773364d, 1.182807992d, $
   0.000206d,     70.328180442d, 5.934076062d, $
   0.000169d,  20995.392966449d, 2.169080622d, $
   0.000191d,  10660.686935042d, 5.405515999d  ]
fbldata = [ fbldata, $
   0.000228d,  33019.021112205d, 4.656985514d, $
   0.000184d,  -4933.208440333d, 3.327476868d, $
   0.000220d,   -135.625325010d, 1.765430262d, $
   0.000166d,  23141.558382925d, 3.454132746d, $
   0.000191d,   6144.558353121d, 5.020393445d, $
   0.000180d,   6084.003848555d, 0.602182191d, $
   0.000163d,  17782.732072784d, 4.960593133d, $
   0.000225d,  16460.333529525d, 2.596451817d, $
   0.000222d,   5905.702242076d, 3.731990323d, $
   0.000204d,    227.476132789d, 5.636192701d, $
   0.000159d,  16737.577236597d, 3.600691544d, $
   0.000200d,   6805.653268085d, 0.868220961d, $
   0.000187d,  11919.140866668d, 2.629456641d, $
   0.000161d,    127.471796607d, 2.862574720d, $
   0.000205d,   6286.666278643d, 1.742882331d, $
   0.000189d,    153.778810485d, 4.812372643d, $
   0.000168d,  16723.350142595d, 0.027860588d, $
   0.000149d,  11720.068865232d, 0.659721876d, $
   0.000189d,   5237.921013804d, 5.245313000d, $
   0.000143d,   6709.674040867d, 4.317625647d, $
   0.000146d,   4487.817406270d, 4.815297007d, $
   0.000144d,   -664.756045130d, 5.381366880d, $
   0.000175d,   5127.714692584d, 4.728443327d, $
   0.000162d,   6254.626662524d, 1.435132069d, $
   0.000187d,  47162.516354635d, 1.354371923d  ]
fbldata = [ fbldata, $
   0.000146d,  11080.171578918d, 3.369695406d, $
   0.000180d,   -348.924420448d, 2.490902145d, $
   0.000148d,    151.047669843d, 3.799109588d, $
   0.000157d,   6197.248551160d, 1.284375887d, $
   0.000167d,    146.594251718d, 0.759969109d, $
   0.000133d,  -5331.357443741d, 5.409701889d, $
   0.000154d,     95.979227218d, 3.366890614d, $
   0.000148d,  -6418.140930027d, 3.384104996d, $
   0.000128d,  -6525.804453965d, 3.803419985d, $
   0.000130d,  11293.470674356d, 0.939039445d, $
   0.000152d,  -5729.506447149d, 0.734117523d, $
   0.000138d,    210.117701700d, 2.564216078d, $
   0.000123d,   6066.595360816d, 4.517099537d, $
   0.000140d,  18451.078546566d, 0.642049130d, $
   0.000126d,  11300.584221356d, 3.485280663d, $
   0.000119d,  10027.903195729d, 3.217431161d, $
   0.000151d,   4274.518310832d, 4.404359108d, $
   0.000117d,   6072.958148291d, 0.366324650d, $
   0.000165d,  -7668.637425143d, 4.298212528d, $
   0.000117d,  -6245.048177356d, 5.379518958d, $
   0.000130d,  -5888.449964932d, 4.527681115d, $
   0.000121d,   -543.918059096d, 6.109429504d, $
   0.000162d,   9683.594581116d, 5.720092446d, $
   0.000141d,   6219.339951688d, 0.679068671d, $
   0.000118d,  22743.409379516d, 4.881123092d  ]
fbldata = [ fbldata, $
   0.000129d,   1692.165669502d, 0.351407289d, $
   0.000126d,   5657.405657679d, 5.146592349d, $
   0.000114d,    728.762966531d, 0.520791814d, $
   0.000120d,     52.596639600d, 0.948516300d, $
   0.000115d,     65.220371012d, 3.504914846d, $
   0.000126d,   5881.403728234d, 5.577502482d, $
   0.000158d, 163096.180360983d, 2.957128968d, $
   0.000134d,  12341.806904281d, 2.598576764d, $
   0.000151d,  16627.370915377d, 3.985702050d, $
   0.000109d,   1368.660252845d, 0.014730471d, $
   0.000131d,   6211.263196841d, 0.085077024d, $
   0.000146d,   5792.741760812d, 0.708426604d, $
   0.000146d,    -77.750543984d, 3.121576600d, $
   0.000107d,   5341.013788022d, 0.288231904d, $
   0.000138d,   6281.591377283d, 2.797450317d, $
   0.000113d,  -6277.552925684d, 2.788904128d, $
   0.000115d,   -525.758811831d, 5.895222200d, $
   0.000138d,   6016.468808270d, 6.096188999d, $
   0.000139d,  23539.707386333d, 2.028195445d, $
   0.000146d,  -4176.041342449d, 4.660008502d, $
   0.000107d,  16062.184526117d, 4.066520001d, $
   0.000142d,  83783.548222473d, 2.936315115d, $
   0.000128d,   9380.959672717d, 3.223844306d, $
   0.000135d,   6205.325306007d, 1.638054048d, $
   0.000101d,   2699.734819318d, 5.481603249d  ]
fbldata = [ fbldata, $
   0.000104d,   -568.821874027d, 2.205734493d, $
   0.000103d,   6321.103522627d, 2.440421099d, $
   0.000119d,   6321.208885629d, 2.547496264d, $
   0.000138d,   1975.492545856d, 2.314608466d, $
   0.000121d,    137.033024162d, 4.539108237d, $
   0.000123d,  19402.796952817d, 4.538074405d, $
   0.000119d,  22805.735565994d, 2.869040566d, $
   0.000133d,  64471.991241142d, 6.056405489d, $
   0.000129d,    -85.827298831d, 2.540635083d, $
   0.000131d,  13613.804277336d, 4.005732868d, $
   0.000104d,   9814.604100291d, 1.959967212d, $
   0.000112d,  16097.679950283d, 3.589026260d, $
   0.000123d,   2107.034507542d, 1.728627253d, $
   0.000121d,  36949.230808424d, 6.072332087d, $
   0.000108d, -12539.853380183d, 3.716133846d, $
   0.000113d,  -7875.671863624d, 2.725771122d, $
   0.000109d,   4171.425536614d, 4.033338079d, $
   0.000101d,   6247.911759770d, 3.441347021d, $
   0.000113d,   7330.728427345d, 0.656372122d, $
   0.000113d,  51092.726050855d, 2.791483066d, $
   0.000106d,   5621.842923210d, 1.815323326d, $
   0.000101d,    111.430161497d, 5.711033677d, $
   0.000103d,    909.818733055d, 2.812745443d, $
   0.000101d,   1790.642637886d, 1.965746028d  ]
fbldata = [ fbldata, $  ;; From end of TDB1NS.F
   0.00065d,    6069.776754d,    4.021194d, $
   0.00033d,     213.299095d,    5.543132d, $
  -0.00196d,    6208.294251d,    5.696701d, $
  -0.00173d,      74.781599d,    2.435900d  ]

i1terms = n_elements(fbldata)/3
; T**1                          
fbldata = [ fbldata, $
 102.156724d,   6283.075849991d, 4.249032005d, $
   1.706807d,  12566.151699983d, 4.205904248d, $
   0.269668d,    213.299095438d, 3.400290479d, $
   0.265919d,    529.690965095d, 5.836047367d, $
   0.210568d,     -3.523118349d, 6.262738348d, $
   0.077996d,   5223.693919802d, 4.670344204d, $
   0.054764d,   1577.343542448d, 4.534800170d, $
   0.059146d,     26.298319800d, 1.083044735d, $
   0.034420d,   -398.149003408d, 5.980077351d, $
   0.032088d,  18849.227549974d, 4.162913471d, $
   0.033595d,   5507.553238667d, 5.980162321d, $
   0.029198d,   5856.477659115d, 0.623811863d, $
   0.027764d,    155.420399434d, 3.745318113d, $
   0.025190d,   5746.271337896d, 2.980330535d, $
   0.022997d,   -796.298006816d, 1.174411803d, $
   0.024976d,   5760.498431898d, 2.467913690d, $
   0.021774d,    206.185548437d, 3.854787540d, $
   0.017925d,   -775.522611324d, 1.092065955d, $
   0.013794d,    426.598190876d, 2.699831988d, $
   0.013276d,   6062.663207553d, 5.845801920d, $
   0.011774d,  12036.460734888d, 2.292832062d, $
   0.012869d,   6076.890301554d, 5.333425680d, $
   0.012152d,   1059.381930189d, 6.222874454d, $
   0.011081d,     -7.113547001d, 5.154724984d, $
   0.010143d,   4694.002954708d, 4.044013795d  ]
fbldata = [ fbldata, $
   0.009357d,   5486.777843175d, 3.416081409d, $
   0.010084d,    522.577418094d, 0.749320262d, $
   0.008587d,  10977.078804699d, 2.777152598d, $
   0.008628d,   6275.962302991d, 4.562060226d, $
   0.008158d,   -220.412642439d, 5.806891533d, $
   0.007746d,   2544.314419883d, 1.603197066d, $
   0.007670d,   2146.165416475d, 3.000200440d, $
   0.007098d,     74.781598567d, 0.443725817d, $
   0.006180d,   -536.804512095d, 1.302642751d, $
   0.005818d,   5088.628839767d, 4.827723531d, $
   0.004945d,  -6286.598968340d, 0.268305170d, $
   0.004774d,   1349.867409659d, 5.808636673d, $
   0.004687d,   -242.728603974d, 5.154890570d, $
   0.006089d,   1748.016413067d, 4.403765209d, $
   0.005975d,  -1194.447010225d, 2.583472591d, $
   0.004229d,    951.718406251d, 0.931172179d, $
   0.005264d,    553.569402842d, 2.336107252d, $
   0.003049d,   5643.178563677d, 1.362634430d, $
   0.002974d,   6812.766815086d, 1.583012668d, $
   0.003403d,  -2352.866153772d, 2.552189886d, $
   0.003030d,    419.484643875d, 5.286473844d, $
   0.003210d,     -7.046236698d, 1.863796539d, $
   0.003058d,   9437.762934887d, 4.226420633d, $
   0.002589d,  12352.852604545d, 1.991935820d, $
   0.002927d,   5216.580372801d, 2.319951253d  ]
fbldata = [ fbldata, $
   0.002425d,   5230.807466803d, 3.084752833d, $
   0.002656d,   3154.687084896d, 2.487447866d, $
   0.002445d,  10447.387839604d, 2.347139160d, $
   0.002990d,   4690.479836359d, 6.235872050d, $
   0.002890d,   5863.591206116d, 0.095197563d, $
   0.002498d,   6438.496249426d, 2.994779800d, $
   0.001889d,   8031.092263058d, 3.569003717d, $
   0.002567d,    801.820931124d, 3.425611498d, $
   0.001803d, -71430.695617928d, 2.192295512d, $
   0.001782d,      3.932153263d, 5.180433689d, $
   0.001694d,  -4705.732307544d, 4.641779174d, $
   0.001704d,  -1592.596013633d, 3.997097652d, $
   0.001735d,   5849.364112115d, 0.417558428d, $
   0.001643d,   8429.241266467d, 2.180619584d, $
   0.001680d,     38.133035638d, 4.164529426d, $
   0.002045d,   7084.896781115d, 0.526323854d, $
   0.001458d,   4292.330832950d, 1.356098141d, $
   0.001437d,     20.355319399d, 3.895439360d, $
   0.001738d,   6279.552731642d, 0.087484036d, $
   0.001367d,  14143.495242431d, 3.987576591d, $
   0.001344d,   7234.794256242d, 0.090454338d, $
   0.001438d,  11499.656222793d, 0.974387904d, $
   0.001257d,   6836.645252834d, 1.509069366d, $
   0.001358d,  11513.883316794d, 0.495572260d, $
   0.001628d,   7632.943259650d, 4.968445721d  ]
fbldata = [ fbldata, $
   0.001169d,    103.092774219d, 2.838496795d, $
   0.001162d,   4164.311989613d, 3.408387778d, $
   0.001092d,   6069.776754553d, 3.617942651d, $
   0.001008d,  17789.845619785d, 0.286350174d, $
   0.001008d,    639.897286314d, 1.610762073d, $
   0.000918d,  10213.285546211d, 5.532798067d, $
   0.001011d,  -6256.777530192d, 0.661826484d, $
   0.000753d,  16730.463689596d, 3.905030235d, $
   0.000737d,  11926.254413669d, 4.641956361d, $
   0.000694d,   3340.612426700d, 2.111120332d, $
   0.000701d,   3894.181829542d, 2.760823491d, $
   0.000689d,   -135.065080035d, 4.768800780d, $
   0.000700d,  13367.972631107d, 5.760439898d, $
   0.000664d,   6040.347246017d, 1.051215840d, $
   0.000654d,   5650.292110678d, 4.911332503d, $
   0.000788d,   6681.224853400d, 4.699648011d, $
   0.000628d,   5333.900241022d, 5.024608847d, $
   0.000755d,   -110.206321219d, 4.370971253d, $
   0.000628d,   6290.189396992d, 3.660478857d, $
   0.000635d,  25132.303399966d, 4.121051532d, $
   0.000534d,   5966.683980335d, 1.173284524d, $
   0.000543d,   -433.711737877d, 0.345585464d, $
   0.000517d,  -1990.745017041d, 5.414571768d, $
   0.000504d,   5767.611978898d, 2.328281115d, $
   0.000485d,   5753.384884897d, 1.685874771d  ]
fbldata = [ fbldata, $
   0.000463d,   7860.419392439d, 5.297703006d, $
   0.000604d,    515.463871093d, 0.591998446d, $
   0.000443d,  12168.002696575d, 4.830881244d, $
   0.000570d,    199.072001436d, 3.899190272d, $
   0.000465d,  10969.965257698d, 0.476681802d, $
   0.000424d,  -7079.373856808d, 1.112242763d, $
   0.000427d,    735.876513532d, 1.994214480d, $
   0.000478d,  -6127.655450557d, 3.778025483d, $
   0.000414d,  10973.555686350d, 5.441088327d, $
   0.000512d,   1589.072895284d, 0.107123853d, $
   0.000378d,  10984.192351700d, 0.915087231d, $
   0.000402d,  11371.704689758d, 4.107281715d, $
   0.000453d,   9917.696874510d, 1.917490952d, $
   0.000395d,    149.563197135d, 2.763124165d, $
   0.000371d,   5739.157790895d, 3.112111866d, $
   0.000350d,  11790.629088659d, 0.440639857d, $
   0.000356d,   6133.512652857d, 5.444568842d, $
   0.000344d,    412.371096874d, 5.676832684d, $
   0.000383d,    955.599741609d, 5.559734846d, $
   0.000333d,   6496.374945429d, 0.261537984d, $
   0.000340d,   6055.549660552d, 5.975534987d, $
   0.000334d,   1066.495477190d, 2.335063907d, $
   0.000399d,  11506.769769794d, 5.321230910d, $
   0.000314d,  18319.536584880d, 2.313312404d, $
   0.000424d,   1052.268383188d, 1.211961766d  ]
fbldata = [ fbldata, $
   0.000307d,     63.735898303d, 3.169551388d, $
   0.000329d,     29.821438149d, 6.106912080d, $
   0.000357d,   6309.374169791d, 4.223760346d, $
   0.000312d,  -3738.761430108d, 2.180556645d, $
   0.000301d,    309.278322656d, 1.499984572d, $
   0.000268d,  12043.574281889d, 2.447520648d, $
   0.000257d,  12491.370101415d, 3.662331761d, $
   0.000290d,    625.670192312d, 1.272834584d, $
   0.000256d,   5429.879468239d, 1.913426912d, $
   0.000339d,   3496.032826134d, 4.165930011d, $
   0.000283d,   3930.209696220d, 4.325565754d, $
   0.000241d,  12528.018664345d, 3.832324536d, $
   0.000304d,   4686.889407707d, 1.612348468d, $
   0.000259d,  16200.772724501d, 3.470173146d, $
   0.000238d,  12139.553509107d, 1.147977842d, $
   0.000236d,   6172.869528772d, 3.776271728d, $
   0.000296d,  -7058.598461315d, 0.460368852d, $
   0.000306d,  10575.406682942d, 0.554749016d, $
   0.000251d,  17298.182327326d, 0.834332510d, $
   0.000290d,   4732.030627343d, 4.759564091d, $
   0.000261d,   5884.926846583d, 0.298259862d, $
   0.000249d,   5547.199336460d, 3.749366406d, $
   0.000213d,  11712.955318231d, 5.415666119d, $
   0.000223d,   4701.116501708d, 2.703203558d, $
   0.000268d,   -640.877607382d, 0.283670793d  ]
fbldata = [ fbldata, $
   0.000209d,   5636.065016677d, 1.238477199d, $
   0.000193d,  10177.257679534d, 1.943251340d, $
   0.000182d,   6283.143160294d, 2.456157599d, $
   0.000184d,   -227.526189440d, 5.888038582d, $
   0.000182d,  -6283.008539689d, 0.241332086d, $
   0.000228d,  -6284.056171060d, 2.657323816d, $
   0.000166d,   7238.675591600d, 5.930629110d, $
   0.000167d,   3097.883822726d, 5.570955333d, $
   0.000159d,   -323.505416657d, 5.786670700d, $
   0.000154d,  -4136.910433516d, 1.517805532d, $
   0.000176d,  12029.347187887d, 3.139266834d, $
   0.000167d,  12132.439962106d, 3.556352289d, $
   0.000153d,    202.253395174d, 1.463313961d, $
   0.000157d,  17267.268201691d, 1.586837396d, $
   0.000142d,  83996.847317911d, 0.022670115d, $
   0.000152d,  17260.154654690d, 0.708528947d, $
   0.000144d,   6084.003848555d, 5.187075177d, $
   0.000135d,   5756.566278634d, 1.993229262d, $
   0.000134d,   5750.203491159d, 3.457197134d, $
   0.000144d,   5326.786694021d, 6.066193291d, $
   0.000160d,  11015.106477335d, 1.710431974d, $
   0.000133d,   3634.621024518d, 2.836451652d, $
   0.000134d,  18073.704938650d, 5.453106665d, $
   0.000134d,   1162.474704408d, 5.326898811d, $
   0.000128d,   5642.198242609d, 2.511652591d  ]
fbldata = [ fbldata, $
   0.000160d,    632.783739313d, 5.628785365d, $
   0.000132d,  13916.019109642d, 0.819294053d, $
   0.000122d,  14314.168113050d, 5.677408071d, $
   0.000125d,  12359.966151546d, 5.251984735d, $
   0.000121d,   5749.452731634d, 2.210924603d, $
   0.000136d,   -245.831646229d, 1.646502367d, $
   0.000120d,   5757.317038160d, 3.240883049d, $
   0.000134d,  12146.667056108d, 3.059480037d, $
   0.000137d,   6206.809778716d, 1.867105418d, $
   0.000141d,  17253.041107690d, 2.069217456d, $
   0.000129d,  -7477.522860216d, 2.781469314d, $
   0.000116d,   5540.085789459d, 4.281176991d, $
   0.000116d,   9779.108676125d, 3.320925381d, $
   0.000129d,   5237.921013804d, 3.497704076d, $
   0.000113d,   5959.570433334d, 0.983210840d, $
   0.000122d,   6282.095528923d, 2.674938860d, $
   0.000140d,    -11.045700264d, 4.957936982d, $
   0.000108d,  23543.230504682d, 1.390113589d, $
   0.000106d, -12569.674818332d, 0.429631317d, $
   0.000110d,   -266.607041722d, 5.501340197d, $
   0.000115d,  12559.038152982d, 4.691456618d, $
   0.000134d,  -2388.894020449d, 0.577313584d, $
   0.000109d,  10440.274292604d, 6.218148717d, $
   0.000102d,   -543.918059096d, 1.477842615d, $
   0.000108d,  21228.392023546d, 2.237753948d  ]
fbldata = [ fbldata, $
   0.000101d,  -4535.059436924d, 3.100492232d, $
   0.000103d,     76.266071276d, 5.594294322d, $
   0.000104d,    949.175608970d, 5.674287810d, $
   0.000101d,  13517.870106233d, 2.196632348d, $
   0.000100d,  11933.367960670d, 4.056084160d  ]

i2terms = n_elements(fbldata)/3
; T**2                          
fbldata = [ fbldata, $
   4.322990d,   6283.075849991d, 2.642893748d, $
   0.406495d,      0.000000000d, 4.712388980d, $
   0.122605d,  12566.151699983d, 2.438140634d, $
   0.019476d,    213.299095438d, 1.642186981d, $
   0.016916d,    529.690965095d, 4.510959344d, $
   0.013374d,     -3.523118349d, 1.502210314d, $
   0.008042d,     26.298319800d, 0.478549024d, $
   0.007824d,    155.420399434d, 5.254710405d, $
   0.004894d,   5746.271337896d, 4.683210850d, $
   0.004875d,   5760.498431898d, 0.759507698d, $
   0.004416d,   5223.693919802d, 6.028853166d, $
   0.004088d,     -7.113547001d, 0.060926389d, $
   0.004433d,  77713.771467920d, 3.627734103d, $
   0.003277d,  18849.227549974d, 2.327912542d, $
   0.002703d,   6062.663207553d, 1.271941729d, $
   0.003435d,   -775.522611324d, 0.747446224d, $
   0.002618d,   6076.890301554d, 3.633715689d, $
   0.003146d,    206.185548437d, 5.647874613d, $
   0.002544d,   1577.343542448d, 6.232904270d, $
   0.002218d,   -220.412642439d, 1.309509946d, $
   0.002197d,   5856.477659115d, 2.407212349d, $
   0.002897d,   5753.384884897d, 5.863842246d, $
   0.001766d,    426.598190876d, 0.754113147d, $
   0.001738d,   -796.298006816d, 2.714942671d, $
   0.001695d,    522.577418094d, 2.629369842d  ]
fbldata = [ fbldata, $
   0.001584d,   5507.553238667d, 1.341138229d, $
   0.001503d,   -242.728603974d, 0.377699736d, $
   0.001552d,   -536.804512095d, 2.904684667d, $
   0.001370d,   -398.149003408d, 1.265599125d, $
   0.001889d,  -5573.142801634d, 4.413514859d, $
   0.001722d,   6069.776754553d, 2.445966339d, $
   0.001124d,   1059.381930189d, 5.041799657d, $
   0.001258d,    553.569402842d, 3.849557278d, $
   0.000831d,    951.718406251d, 2.471094709d, $
   0.000767d,   4694.002954708d, 5.363125422d, $
   0.000756d,   1349.867409659d, 1.046195744d, $
   0.000775d,    -11.045700264d, 0.245548001d, $
   0.000597d,   2146.165416475d, 4.543268798d, $
   0.000568d,   5216.580372801d, 4.178853144d, $
   0.000711d,   1748.016413067d, 5.934271972d, $
   0.000499d,  12036.460734888d, 0.624434410d, $
   0.000671d,  -1194.447010225d, 4.136047594d, $
   0.000488d,   5849.364112115d, 2.209679987d, $
   0.000621d,   6438.496249426d, 4.518860804d, $
   0.000495d,  -6286.598968340d, 1.868201275d, $
   0.000456d,   5230.807466803d, 1.271231591d, $
   0.000451d,   5088.628839767d, 0.084060889d, $
   0.000435d,   5643.178563677d, 3.324456609d, $
   0.000387d,  10977.078804699d, 4.052488477d, $
   0.000547d, 161000.685737473d, 2.841633844d  ]
fbldata = [ fbldata, $
   0.000522d,   3154.687084896d, 2.171979966d, $
   0.000375d,   5486.777843175d, 4.983027306d, $
   0.000421d,   5863.591206116d, 4.546432249d, $
   0.000439d,   7084.896781115d, 0.522967921d, $
   0.000309d,   2544.314419883d, 3.172606705d, $
   0.000347d,   4690.479836359d, 1.479586566d, $
   0.000317d,    801.820931124d, 3.553088096d, $
   0.000262d,    419.484643875d, 0.606635550d, $
   0.000248d,   6836.645252834d, 3.014082064d, $
   0.000245d,  -1592.596013633d, 5.519526220d, $
   0.000225d,   4292.330832950d, 2.877956536d, $
   0.000214d,   7234.794256242d, 1.605227587d, $
   0.000205d,   5767.611978898d, 0.625804796d, $
   0.000180d,  10447.387839604d, 3.499954526d, $
   0.000229d,    199.072001436d, 5.632304604d, $
   0.000214d,    639.897286314d, 5.960227667d, $
   0.000175d,   -433.711737877d, 2.162417992d, $
   0.000209d,    515.463871093d, 2.322150893d, $
   0.000173d,   6040.347246017d, 2.556183691d, $
   0.000184d,   6309.374169791d, 4.732296790d, $
   0.000227d, 149854.400134205d, 5.385812217d, $
   0.000154d,   8031.092263058d, 5.120720920d, $
   0.000151d,   5739.157790895d, 4.815000443d, $
   0.000197d,   7632.943259650d, 0.222827271d, $
   0.000197d,     74.781598567d, 3.910456770d  ]
fbldata = [ fbldata, $
   0.000138d,   6055.549660552d, 1.397484253d, $
   0.000149d,  -6127.655450557d, 5.333727496d, $
   0.000137d,   3894.181829542d, 4.281749907d, $
   0.000135d,   9437.762934887d, 5.979971885d, $
   0.000139d,  -2352.866153772d, 4.715630782d, $
   0.000142d,   6812.766815086d, 0.513330157d, $
   0.000120d,  -4705.732307544d, 0.194160689d, $
   0.000131d, -71430.695617928d, 0.000379226d, $
   0.000124d,   6279.552731642d, 2.122264908d, $
   0.000108d,  -6256.777530192d, 0.883445696d  ]

i3terms = n_elements(fbldata)/3
; T**3                          
fbldata = [ fbldata, $
   0.143388d,   6283.075849991d, 1.131453581d, $
   0.006671d,  12566.151699983d, 0.775148887d, $
   0.001480d,    155.420399434d, 0.480016880d, $
   0.000934d,    213.299095438d, 6.144453084d, $
   0.000795d,    529.690965095d, 2.941595619d, $
   0.000673d,   5746.271337896d, 0.120415406d, $
   0.000672d,   5760.498431898d, 5.317009738d, $
   0.000389d,   -220.412642439d, 3.090323467d, $
   0.000373d,   6062.663207553d, 3.003551964d, $
   0.000360d,   6076.890301554d, 1.918913041d, $
   0.000316d,    -21.340641002d, 5.545798121d, $
   0.000315d,   -242.728603974d, 1.884932563d, $
   0.000278d,    206.185548437d, 1.266254859d, $
   0.000238d,   -536.804512095d, 4.532664830d, $
   0.000185d,    522.577418094d, 4.578313856d, $
   0.000245d,  18849.227549974d, 0.587467082d, $
   0.000180d,    426.598190876d, 5.151178553d, $
   0.000200d,    553.569402842d, 5.355983739d, $
   0.000141d,   5223.693919802d, 1.336556009d, $
   0.000104d,   5856.477659115d, 4.239842759d  ]

i4terms = n_elements(fbldata)/3
; T**4                          
fbldata = [ fbldata, $
   0.003826d,   6283.075849991d, 5.705257275d, $
   0.000303d,  12566.151699983d, 5.407132842d, $
   0.000209d,    155.420399434d, 1.989815753d  ]

    nterms = n_elements(fbldata)/3
    fbldata = reform(fbldata, 3, nterms, /overwrite)
    const0 = reform(fbldata[0,*], nterms)
    freq0  = reform(fbldata[1,*], nterms)
    phase0 = reform(fbldata[2,*], nterms)

    texp = dblarr(nterms) +   0
    texp[i1terms:i2terms-1] = 1
    texp[i2terms:i3terms-1] = 2
    texp[i3terms:i4terms-1] = 3
    texp[i4terms:*        ] = 4

  endif

  if n_elements(tbase) EQ 0 then tbase = 0D
  t = ((tbase[0]-2451545D) + jd[0])/365250.0D
  if t EQ 0 then t = 1d-100

  ph = freq0 * t + phase0 
  sint = sin( ph )
  sinf = const0 * t^texp

  dt = total(sinf*sint)*1d-6
  if arg_present(deriv) then $
    deriv = total(sinf*(texp*sint/t + freq0*cos(ph)))*(1d-6/365250.0D)

  return, dt
end

function tdb2tdt, jd, deriv=deriv, tbase=tbase

  sz = size(jd)
  if sz[0] EQ 0 then $
    return, tdb2tdt_calc(jd, deriv=deriv, tbase=tbase)

  result = reform(double(jd), sz[1:sz[0]])
  if arg_present(deriv) then begin
      deriv = reform(double(jd), sz[1:sz[0]])
      for i = 0L, sz[sz[0]+2]-1 do begin
          result[i] = tdb2tdt_calc(jd[i], deriv=dd, tbase=tbase)
          deriv[i] = dd
      endfor
  endif else begin
      for i = 0L, sz[sz[0]+2]-1 do begin
          result[i] = tdb2tdt_calc(jd[i], tbase=tbase)
      endfor
  endelse

  return, result
end

