PRO makemap
; makemap IDL script.
; Version history:
; v 1.2 25 AUG 2011 / Balt Indermuehle balt.indermuehle@csiro.au - Added Astrolib and Coyote download links and installation instructions
; v 1.1 03 JUL 2011 / Balt Indermuehle balt.indermuehle@csiro.au - Modified to produce coloured plots
; v 1.0 02 JUL 2011 / Balt Indermuehle balt.indermuehle@csiro.au - Created
;
; This example script assumes you have livedata, gridzilla, kvis, miriad and IDL 7 or higher installed.
; Further, the IDL AstroLib is required (free download) and Coyote's new graphics library (free download also):
; Astro library: Download from here http://www.narrabri.atnf.csiro.au/mopra/astron.dir.zip
; Coyote graphics library for IDL: Download from here: http://www.idlcoyote.com/graphics_tips/coyote_graphics.php"
; To install the Astro library:
; Create a directory ~/IDL in your machine home (if it doesn't exist already)
; Unpack astron.dir.zip into that directory so there's ~/IDL/astron.dir/...
; Edit the file <code>~/IDL/idl_startup</code> and add the following two lines (the compile option is not strictly required for astrolib, but it's a good practise to have 32bit integers by default in the 21st century so your integer loop variables won't be +/-16k cyclic....):
;  astrolib
;  Compile_Opt DEFINT32
;
; This is a sample IDL script to semi automatically create noise- and emission maps for Mopra data.
; First, prepare your data as follows:
; 1.) Run livedata. Note that you are limited to do one IF at a time if you do it manually:
;     > In configuration, select MOPRA
;     > Select Bandpass calibration
;     > Select Write data
;     > in "Input Data Selection", select only IF1 (by default IF1 and IF2 are selected)
;     > set Channel Range to go from 1 to 4096 (if Mopra Zoom or Fastmapping data) or 1 to 8192 (if Mopra Broadband data)
;     > in Bandpass Calibration, set the Mask to go from 1 to 300 and from 3796 to 4096 
;       (or 7892 to 8192 for Broadband mode). This cuts off the top and bottom 300 channels
;     > Select all your data files 
;     > Enqueue all files and hit START. This will take a few minutes to complete.
;     > you now should have a *.sdfits file for every *.rpf file you gave it to process
;
; 2.) Run gridzilla on the files created in step 1:
;     > Select MOPRA from the presets (top right)
;     > Select the velocity range to the range you are expecting (or leave at default -1000 to +1000)
;     > Set your line rest frequency. This is where the center of the Z axis in the data cube will be.
;     > Adjust the Beam FWHM to the size that corresponds to your observing frequency
;     > Set the output FITS filename. In this example N2H+, or "n2hp" for the filename.
;     > Press GO. Get a coffee. When you get back, you'll have a file called "n2hp_MEAN.fits" which has the gridded map inside.
;
; 3.) Now you need to create the 0th moment map. 
;     > Start kvis. You first need to locate which channels contain your emission:
;     > Open the n2hp_MEAN.fits file. Select View -> Profile Mode: Line
;     > Move the mouse in the map until you find your emission. Now you can use the Movie mode to play through your data. Locate the channel number where your line emission starts and where it ends. In the example below, this was found to be from 2976 to 3123.
;     > While you're at it, you also want to find an emission free region for your noise map. In the example below, that is from 500 to 2500.
;     > Now start miriad and create the actual moment map. From the miriad command line, type these commands:
;     > fits in=n2hp_MEAN.fits op=xyin out=IF1
;     > imsub in=IF1 out=n2hp region='images(2976,3123)'
;     > moment in=n2hp out=n2hp.mom0 mom=0
;     > Now open the n2hp.mom0 map with kvis and save that file as "n2hp_hdr.fits". This file will be needed to read header information that otherwise is difficult to create from scratch and is needed to display your data further down in this IDL script.
;     > n2hp.mom0 (or the FITS file you just saved) essentially contains your science result. However, to make it look pretty, we now need to run this IDL script.
; 4.) Run this IDL script.
;     > First, change the information below where it says YOUR INPUT GOES HERE. 
;     > compile and run the noise map (set noise_plot = 1)
;     > compile and run the emission map (set noise_plot = 0)
;     > Experiment with fit_baseline! Your maps often can be made much smoother by fitting a baseline along the Z (spectral) axis!
;     > read through the source so you understand what happens where and why! Use this as a template to extend and build your own!
;     > I have tried to comment all that I think needs explaining. If you have problems or praise, feel free to contact me at the email above.
; END
;========================================================================
; YOUR INPUT GOES HERE:

title      = 'N2H+ Map'          ; The title for your map
fullpath   = '~/Documents/Mopra/M581/'     ; The full path leading to the data files below. MUST have trailing slash!
filename   = 'n2hp_MEAN.fits'    ; The filename for the input data (you created this above)
headerfile = 'n2hp_hdr.fits'     ; The filename for the FITS header 0th moment map (you created this above)

; SET TO YOUR VALUES FOUND ABOVE
noise_from_channel  = 500
noise_to_channel    = 2500
signal_from_channel = 2976
signal_to_channel   = 3123

show_beam    = 1          ; Set to 0 if you don't want the beam displayed.
pix_discard  = 3          ; How many pixels to discard on each side
fit_baseline = 0          ; 0=don't fit one, 1=do fit
noise_plot   = 0          ; 0=plot a emission plot (changes the axis labels)
output       = 'n2hp_map' ; The filename for the output. Note that 'emission' and 'noise' gets added automatically
num_cont     = 10         ; Number of contours to use for the contour plot
num_colours  = 60         ; Number of colours to use in the colour table
colourtable  = 15		      ; Which colour table to use. Some colours are (descriptions are for Brewer colour schemes, in reverse):
                          ; 0: dark green to white
                          ; 1: dark blue - turquoise - eggshell
                          ; 2: light blue - turquoise - eggshell
                          ; 3: dark green to white
                          ; 4: dark green - blue - white
                          ; 5: dark blue to white
                          ; 6: purple to white
                          ; 7: purple - eggshell - white
                          ; 8: red - pink - white
                          ; 9: red - white
                          ; 10: brown - orange - eggshell
                          ; 11: red - orange - eggshell
                          ; 12: purple - white
                          ; 13: blue - white
                          ; 14: green - white
                          ; 15: red - orange - white
                          ; 17: black - white
                          ; 18: purple - white - brown
                          ; 19: green - white - brown
                          ; 20: green - white - purple
                          ; 21: light green - white - purple
                          ; 22: blue - white - red
                          ; 23: black - white - red
                          ; 24: light blue - white - red
                          ; 25: light blue - eggshell - red
                          ; 26: light green - eggshell - red
                          ; 27: orange - green - purple
                          ; 28: white - orange - blue - green - black

; END OF YOUR INPUT.
;========================================================================


; Set up the graphics environment:
!p.multi = [0, 1, 1 ]  ; plot 1 x 1 plots on one sheet. Set to 0, 2, 2 if you want 2 x 2 plots for example
SET_PLOT, 'PS'
IF noise_plot EQ 1 THEN fname = output + '_noise.ps' ELSE fname = output + '_emission.ps'
DEVICE, filename=fname, XSIZE=22, YSIZE=22, /COLOR, BITS=8, /ENCAPSULATED, FONT_SIZE=8
TVLCT, INDGEN(256), INDGEN(256), INDGEN(256)
pos = [0.1, 0.1, 0.9, 0.9]

data_cube=READFITS(fullpath+filename,HDR_DATA)
data_mom=READFITS(fullpath+headerfile,HDR_MOM)
beam_maj=sxpar(HDR_DATA,'BMAJ')
beam_min=sxpar(HDR_DATA,'BMIN')
beam_pa=sxpar(HDR_DATA,'BPA')
pix_width=sxpar(HDR_DATA,'NAXIS1')
pix_height=sxpar(HDR_DATA,'NAXIS2')

IF noise_plot eq 0 THEN BEGIN
  from_channel = signal_from_channel
  to_channel = signal_to_channel
ENDIF ELSE BEGIN
  from_channel = noise_from_channel
  to_channel = noise_to_channel
ENDELSE

; The entire data cube is now in the variable data_cube. To inspect the data cube
; uncomment the following lines up to and including STOP:
; HELP, data_cube
; DEVICE,/CLOSE
; STOP

; If you want to plot a single spectrum at a given x,y position (default is center pixel) 
; of your map, uncomment the following lines up to and including STOP:
;!p.multi=0
;SET_PLOT,'X'
;WINDOW,1
;PLOT,data_cube(pix_width/2,pix_height/2,*)
;DEVICE,/CLOSE
;STOP

; Now select the data to create a moment map from
PRINT,'Data selection limited to channels ', from_channel, ' to ', to_channel
data = data_cube(pix_width/2, pix_height/2, from_channel:to_channel)
result = MOMENT(data, /double, MDEV=out1, SDEV=out2)

PRINT,'Statistics on the selected data: '
PRINT,'mean :',result[0], ' Variance :',result[1], ' Skewness: ',result[2], 'Kurtosis: ',result[3], ' MDEV: ',out1, ' SDEV: ',out2 
PRINT,'SDev at center: ',out2

sdevc = STRING(out2, Format='(D0.2)')

; Now create the map
data_map = FLTARR(pix_width, pix_height)
rms = FLOAT(0.0)
cnt = LONG(0)
FOR i = pix_discard, pix_width-pix_discard - 1 DO BEGIN
	FOR j = pix_discard, pix_height-pix_discard - 1 DO BEGIN
		;make sure we don't include edge data where we have no data points due to the gridding
		bad = WHERE(FINITE(data_map) EQ 0, count)
		IF (count GT 0) THEN data_map(bad) = 0.0
		data = data_cube(i, j, from_channel:to_channel)
		; obtain the first rms estimate
		result = MOMENT(data, /DOUBLE, MDEV=out1, SDEV=rms1)

		IF (fit_baseline eq 1) THEN BEGIN
			; prepare for the poly fit
			; get the x and y values of the spectrum into a 1d array each
			data = REFORM(data)
			x_num = N_ELEMENTS(data)
			x_axis = FINDGEN(x_num)
			; fit a polynomial on this spectrum
			coeff = ROBUST_POLY_FIT(x_axis,data, 4, yfit, resid)
			data = data-yfit
			result = MOMENT(data, /DOUBLE, MDEV=out1, SDEV=rms2)
		ENDIF ELSE BEGIN
			rms2 = rms1
		ENDELSE

    IF FINITE(rms1) AND FINITE(rms2) THEN BEGIN
			; only apply the fitted data if the rms got better
			IF (rms2 LT rms1) THEN BEGIN
				bestrms = rms2
			ENDIF ELSE BEGIN
				bestrms = rms1
			ENDELSE
			data_map(i, j) = bestrms
			rms = rms + bestrms
			cnt = cnt + 1
		ENDIF ;if finite
	ENDFOR
ENDFOR

;determine the overall RMS from this noisemap subset of data:
rms = rms / cnt
PRINT,'SDev Map: ', rms
sdevt = STRING(rms, Format='(D0.2)')
	
; extract the peak location so we can put a mark there:
pkx1 = -1
pky1 = -1
pkx = 0
pky = 0
FOR x=0,PIX_WIDTH-1 DO BEGIN
  FOR y=0,PIX_HEIGHT-1 DO BEGIN
    IF data_map[x,y] EQ MAX(data_map) THEN BEGIN
      IF pkx EQ x - 1 THEN BEGIN
        ; we need to average, the peak is a plateau
        IF pkx1 GE 0 THEN BEGIN
          ; average from the first plateau x to the current one
          pkx = (pkx1 + x) / 2
        ENDIF ELSE BEGIN
          ; the plateau is new, this is the first value to be set
          pkx1 = x
          pkx = x
        ENDELSE
      ENDIF ELSE BEGIN
          pkx = x        
      ENDELSE
     
      IF pky EQ y - 1 THEN BEGIN
        ; we need to average, the peak is a plateau
        IF pky1 GE 0 THEN BEGIN
          ; average from the first plateau x to the current one
          pky = (pky1 + y) / 2
        ENDIF ELSE BEGIN
          ; the plateau is new, this is the first value to be set
          pky1 = y
          pky = y
        ENDELSE
      ENDIF ELSE BEGIN
        pky = y
      ENDELSE
    ENDIF
  ENDFOR
ENDFOR

; If it's a noise map, we're not using the peak position but just the center position
IF noise_plot EQ 1 THEN BEGIN
  pkx = pix_width / 2.0
  pky = pix_height / 2.0
ENDIF ELSE BEGIN
  PRINT, "Found Peak X,Y:", pkx, pky
ENDELSE



;; Scale the map to peak:
;data_map_scaled = data_map / MAX(data_map) * 255.

;; to remove black edges from underneath the axes:
;FOR pd=0,pix_discard-1 DO BEGIN
;  data_map_scaled[pd,*] = MAX(data_map_scaled)
;  data_map_scaled[*,pd] = MAX(data_map_scaled)
;  data_map_scaled[*,PIX_HEIGHT-(pd+1)] = MAX(data_map_scaled)
;  data_map_scaled[PIX_WIDTH-(pd+1),*] = MAX(data_map_scaled)
;ENDFOR

; Extract the sub image:
IF pix_discard GT 0 THEN HEXTRACT, data_map, HDR_MOM, pix_discard, pix_width-pix_discard-1, pix_discard, pix_height-pix_discard-1

cgLoadCT, colourtable, RGB_Table=pal, CLIP=[1, MAX(data_map)/MAX(data_map)*255], /BREWER, /REVERSE

cp = STRING(MAX(data_map), Format='(D0.2)')

; Create the title
IF noise_plot EQ 1 THEN BEGIN
  title = title + ' Centre RMS: ' + sdevc + 'K / Avg RMS: ' + sdevt + 'K'
ENDIF ELSE BEGIN
  title = title + ' Peak Ta*: ' + cp + 'K'
ENDELSE

; Plot the image and axes
IMCONTOUR, data_map, HDR_MOM, TITLE=title, NLEVELS=num_colours, /FILL, POSITION=pos, PALETTE=pal, TYPE=1, CHARSIZE=1, /OVERLAY

x = FINDGEN(N_ELEMENTS(data_map(*,0)))
y = FINDGEN(N_ELEMENTS(data_map(0,*)))

; add contours
CONTOUR, data_map, x, y, levels=FINDGEN(num_cont) * (cp / (num_cont-1.0D)) + (cp / 2), /FOLLOW, /OVERPLOT

; Add the beam to he lefthand corner
IF show_beam EQ 1 THEN BEGIN
  ICXY2AD, 2, 2, HDR_MOM, ra2000, dec2000
  ICELLIPSE_OPLOT, HDR_MOM, ra2000, dec2000, [beam_maj/2.], [beam_min/2.], [beam_pa], _extra=e, color=0, thick=3
  beamsize = STRING(beam_maj * 3600, FORMAT='(I0)')
  XYOUTS, 2, 2, ALIGNMENT=0.45, 'Beam size ' + beamsize + '"', CHARSIZE=1, COLOR=cgColor("black")
ENDIF

; Add a cross to the peak emission
psym=1
OPLOT,[pkx], [pky], psym=psym, symsize=3, color=255, thick=4
OPLOT,[pkx], [pky], psym=psym, symsize=3, color=0, thick=1.5
	
; Add the color bar
IF noise_plot eq 1 THEN BEGIN
  title = 'RMS per Channel [K]'
ENDIF ELSE BEGIN
  title='Ta* [K]'
ENDELSE

pos=[0.92, 0.1, 0.94, 0.9]
cgColorbar, Title=title, Divisions=12, NColors=num_colours, $
	Range=[MIN(data_map),MAX(data_map)], TickLen=1.0, POSITION=pos, Charsize=1, $
	PALETTE=pal, /VERTICAL, /RIGHT, FORMAT='(D0.2)'

;COLORBAR, POSITION=pos,MAX=MAX(data_map),MIN=MIN(data_map), /VERTICAL, $
;   NCOLORS=250, /TOP, TITLE=title, /RIGHT, DIVISIONS=6, CHARSIZE=1.25
		
DEVICE,/close

PRINT, 'Done!'
END
